<?php

namespace App\Repository;

use App\Http\Resources\Flight;
use App\Http\Resources\FlightCollection;
use GuzzleHttp\Client;
use Symfony\Component\HttpKernel\Exception\HttpException;
use Throwable;

/**
 * FlightRepository
 *
 * @package App\Repository
 */
class FlightRepository
{
    /**
     * @var Client
     */
    private $client;

    /**
     * FlightRepository
     */
    public function __construct()
    {
        $this->client = new Client(
            [
                'base_uri' => env('123MILHAS_API_URL')
            ]
        );
    }

    /**
     * @param string $uri
     * @return mixed
     */
    private function getUri(string $uri)
    {
        try {
            sleep(env('DELAY_IN_SECONDS_123MILHAS_API_URL', 0));

            $response = $this->client->get($uri);
            $data = json_decode($response->getBody(), true);

            return $data;
        } catch (Throwable $e) {
            throw new HttpException(
                $e->getCode(),
                'We encounter some problems while trying to communicate with 123Milhas API. Try again later!'
            );
        }
    }

    public function getAllFlights()
    {
        $response = $this->getUri('flights');

        return new FlightCollection($response);
    }

    public function getAllInboundFlights()
    {
        $response = $this->getUri('flights?inbound=1');

        return new FlightCollection($response);
    }

    public function getAllOutboundFlights()
    {
        $response = $this->getUri('flights?outbound=1');

        return new FlightCollection($response);
    }

    public function getAllGroupsFlights()
    {
        // all flights
        $allFlightsCollection = $this->getAllFlights();
        $allFlightsCollectionSorted = $allFlightsCollection->sortBy('price', SORT_NUMERIC)->values();

        // all outbound flights
        $filteredOutboundFlightsCollection = $allFlightsCollectionSorted->filter(function ($item) {
            return $item->isOutbound();
        })->values();

        $groupedFilteredOutboundFlightsCollection = $filteredOutboundFlightsCollection->groupBy(function ($item) {
            return sprintf('%s-%s-%s', $item['fare'], $item['price'], 'OUT');
        });

        $totalUniqueFlights = $allFlightsCollection->count();
        $cheapestPrice = 0;

        $result = [
            'flights' => json_decode($allFlightsCollectionSorted, true),
            'groups' => [],
            'totalFlights' => $totalUniqueFlights,
        ];

        foreach($groupedFilteredOutboundFlightsCollection as $groupOutbound){
            $totalPriceGroup = 0;
            $priceOutbound = 0;
            $hasOutbound = false;
            $fare = '';

            $flightOutbound = $groupOutbound->first();
            if ($flightOutbound instanceof Flight) {
                $hasOutbound = true;
                $priceOutbound = $flightOutbound['price'];
                $fare = $flightOutbound['fare'];
            }

            // loop through inbound groups
            $filteredInboundFlightsCollection = $allFlightsCollectionSorted->filter(function ($item) use ($fare) {
                return ($item->isInbound() && $item->isFare($fare));
            })->values();

            $groupedFilteredInboundFlightsCollection = $filteredInboundFlightsCollection->groupBy(function ($item) {
                return sprintf('%s-%s-%s', $item['fare'], $item['price'], 'IN');
            });

            foreach($groupedFilteredInboundFlightsCollection as $groupFlightInbound){
                $hasInbound = false;
                $priceInbound = 0;

                $flightInbound = $groupFlightInbound->first();
                if ($flightInbound instanceof Flight) {
                    $hasInbound = true;
                    $priceInbound = $flightInbound['price'];
                }

                if ($hasInbound && $hasOutbound) {
                    $totalPriceGroup = $priceOutbound + $priceInbound;

                    if ($totalPriceGroup < $cheapestPrice || $cheapestPrice == 0) {
                        $cheapestPrice = $totalPriceGroup;
                    }

                    $result['groups'][] = [
                        'totalPrice' => $totalPriceGroup,
                        'outbound' => $groupOutbound,
                        'inbound' => $groupFlightInbound,
                    ];
                }
            }

        }

        // order groups by totalPrice
        usort($result['groups'], function($a, $b) {
            return $a['totalPrice'] <=> $b['totalPrice'];
        });

        // set unique identifer
        $uniqueGroupCounter = 0;
        foreach ($result['groups'] as &$gp) {
            $uniqueGroupCounter++;
            $gp['uniqueId'] = $uniqueGroupCounter;
        }

        $result['totalGroups'] = $uniqueGroupCounter;
        $result['cheapestPrice'] = $cheapestPrice;
        $result['cheapestGroup'] = ($uniqueGroupCounter > 0 ? 1 : 0);

        return $result;
    }
}
